/*
 * Decompiled with CFR 0.152.
 */
package com.snowMirror.upgrade;

import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import org.apache.commons.io.FileUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class H2Driver {
    private final Logger logger = LoggerFactory.getLogger(this.getClass());
    private final Path installationDir;
    private final Path newFilesDir;
    private final String H2_FILE_PREFIX = "h2";
    private final String H2_V1_FILE_PREFIX = "h2-1";
    private final String H2_FILE_PREFIX_IN_WAR = "WEB-INF/lib/h2-";

    public H2Driver(Path installationDir, Path newFilesDir) {
        this.installationDir = installationDir;
        this.newFilesDir = newFilesDir;
    }

    public void handle() {
        try {
            boolean preserved = this.preserveOldH2DriverIfNecessary();
            if (!preserved) {
                this.extractH2v2DriverIfNecessary();
            }
        }
        catch (Exception e) {
            this.logger.error("Failed to handle H2 driver.", e);
        }
    }

    private boolean preserveOldH2DriverIfNecessary() throws IOException {
        int currentH2Format = this.h2FormatVersion().orElse(Integer.MAX_VALUE);
        if (currentH2Format >= 3) {
            return false;
        }
        Path driverDirPath = this.installationDir.resolve("snow-mirror").resolve("data").resolve("driver");
        if (currentH2Format == 1 || currentH2Format == 2) {
            Set<Path> driverFiles;
            Set<Path> set = driverFiles = currentH2Format == 1 ? this.getH2v1DriverFiles(driverDirPath) : this.getH2DriverFiles(driverDirPath);
            if (!driverFiles.isEmpty()) {
                this.logger.info("H2 driver is already present. Nothing else to do.");
                return true;
            }
            this.extractDriverFromCurrentlyInstalledWar();
        }
        if (!this.getH2DriverFiles(driverDirPath).isEmpty()) {
            this.logger.info("H2 driver extracted.");
            return true;
        }
        this.logger.info("H2 driver not present.");
        return false;
    }

    private void extractH2v2DriverIfNecessary() throws IOException {
        Path driverDirPath = this.installationDir.resolve("snow-mirror").resolve("data").resolve("driver");
        Set<Path> h2v1Drivers = this.getH2v1DriverFiles(driverDirPath);
        if (!h2v1Drivers.isEmpty()) {
            this.logger.info("Not copying H2v2 driver. Using H2v1 to stay backward compatible.");
            return;
        }
        if (Files.exists(driverDirPath, new LinkOption[0])) {
            try (Stream<Path> stream = Files.list(driverDirPath);){
                stream.filter(it -> it.getFileName() != null && it.getFileName().startsWith("h2-2")).forEach(it -> {
                    try {
                        this.logger.info("Deleting old H2 v2 driver.");
                        Files.delete(it);
                        this.logger.info("Deleted old H2 v2 driver.");
                    }
                    catch (IOException e) {
                        this.logger.info("Failed to delete " + it, e);
                    }
                });
            }
        }
        Path h2v2JarPath = this.newFilesDir.resolve("snow-mirror").resolve("installation").resolve("h2.jar");
        if (!Files.exists(driverDirPath, new LinkOption[0])) {
            FileUtils.forceMkdir(driverDirPath.toFile());
        }
        Path h2FilePath = driverDirPath.resolve("h2.jar");
        this.logger.info("Copying a new version of H2 v2 driver from {} to {}", (Object)h2v2JarPath, (Object)h2FilePath);
        FileUtils.copyFile(h2v2JarPath.toFile(), h2FilePath.toFile());
        this.logger.info("Copied a new version of H2 v2 driver.");
    }

    private Set<Path> getH2v1DriverFiles(Path driverDirPath) throws IOException {
        return this.getH2DriverFiles(driverDirPath, "h2-1");
    }

    private Set<Path> getH2DriverFiles(Path driverDirPath) throws IOException {
        return this.getH2DriverFiles(driverDirPath, "h2");
    }

    private Set<Path> getH2DriverFiles(Path driverDirPath, String fileDriverNamePrefix) throws IOException {
        Set<Path> h2v1Drivers = Set.of();
        if (Files.exists(driverDirPath, new LinkOption[0])) {
            try (Stream<Path> stream = Files.list(driverDirPath);){
                h2v1Drivers = stream.filter(it -> it.getFileName() != null && it.getFileName().toString().startsWith(fileDriverNamePrefix)).collect(Collectors.toSet());
            }
        }
        return h2v1Drivers;
    }

    private void extractDriverFromCurrentlyInstalledWar() throws IOException {
        Path warFilePath = this.installationDir.resolve("snow-mirror").resolve("webapp").resolve("snowMirror.war");
        Path targetDirPath = this.installationDir.resolve("snow-mirror").resolve("data").resolve("driver");
        this.logger.info("Extracting H2 driver from {} to {}", (Object)warFilePath, (Object)targetDirPath);
        if (!Files.exists(targetDirPath, new LinkOption[0])) {
            FileUtils.forceMkdir(targetDirPath.toFile());
        }
        this.extractFiles(warFilePath, "WEB-INF/lib/h2-", targetDirPath);
        this.logger.info("Extraction finished.");
    }

    public Optional<Integer> h2FormatVersion() throws IOException {
        this.logger.info("Checking, whether a current installation uses H2 version 1.");
        Optional<Integer> format = Optional.empty();
        Path snowMirrorPropertiesPath = this.installationDir.resolve("snowMirror.properties");
        List<String> lines = Files.readAllLines(snowMirrorPropertiesPath);
        boolean snowMirrorPropertiesUsesH2 = lines.stream().anyMatch(line -> line.matches(".*config\\.jdbc\\.url\\s+=\\s+jdbc:h2:.*"));
        if (snowMirrorPropertiesUsesH2) {
            Path oldH2FileH2DbPath = this.installationDir.resolve("snow-mirror").resolve("data").resolve("h2").resolve("snowmirror.h2.db");
            if (Files.exists(oldH2FileH2DbPath, new LinkOption[0])) {
                format = Optional.of(1);
            } else {
                Path oldH2FileMvDbPath = this.installationDir.resolve("snow-mirror").resolve("data").resolve("h2").resolve("snowmirror.mv.db");
                if (Files.exists(oldH2FileMvDbPath, new LinkOption[0])) {
                    format = Optional.of(this.getFormatVersion(oldH2FileMvDbPath));
                }
            }
        }
        this.logger.info("Uses H2: {}", (Object)format.map(it -> "yes").orElse("no"));
        this.logger.info("H2 format: {}", (Object)format.map(Object::toString).orElse("-"));
        return format;
    }

    public Integer getFormatVersion(Path h2File) throws IOException {
        return Files.lines(h2File).findFirst().map(line -> Integer.valueOf(line.replaceFirst(".*format:", "").replaceFirst(",.*", ""))).orElseThrow();
    }

    public void extractFiles(Path zipFilePath, String fileToExtractPrefix, Path targetDirPath) {
        try (ZipInputStream zis = new ZipInputStream(new FileInputStream(zipFilePath.toFile()));){
            ZipEntry zipEntry = zis.getNextEntry();
            while (zipEntry != null) {
                if (zipEntry.getName().startsWith(fileToExtractPrefix)) {
                    Path newFilePath;
                    String fileName = zipEntry.getName();
                    int i = zipEntry.getName().lastIndexOf("/");
                    if (i != -1) {
                        fileName = fileName.substring(i + 1);
                    }
                    if (!Files.exists(newFilePath = targetDirPath.resolve(fileName), new LinkOption[0])) {
                        Files.createFile(newFilePath, new FileAttribute[0]);
                    }
                    try (FileOutputStream fos = new FileOutputStream(newFilePath.toFile());){
                        int len;
                        byte[] buffer = new byte[4096];
                        while ((len = zis.read(buffer)) > 0) {
                            fos.write(buffer, 0, len);
                        }
                    }
                    this.logger.info("Extracted a file from {} to {}", (Object)zipFilePath, (Object)newFilePath);
                }
                zipEntry = zis.getNextEntry();
            }
        }
        catch (IOException e) {
            e.printStackTrace();
        }
    }
}

