/*
 * Decompiled with CFR 0.152.
 */
package com.snowMirror.upgrade;

import com.snowMirror.upgrade.H2Driver;
import com.snowMirror.upgrade.Keystore;
import com.snowMirror.upgrade.ServerXml;
import com.snowMirror.upgrade.TomcatBinary;
import com.snowMirror.upgrade.WebXml;
import java.io.File;
import java.io.FileFilter;
import java.io.IOException;
import java.net.URLDecoder;
import java.nio.charset.StandardCharsets;
import java.nio.file.CopyOption;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.commons.io.FileUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Upgrade {
    private static Logger logger = null;

    public static void main(String[] args) {
        try {
            System.setProperty("snowMirror.logPath", Upgrade.getLogPath(args));
            System.setProperty("log4j2.configurationFile", "log4j2-upgrade.xml");
            logger = LoggerFactory.getLogger(Upgrade.class);
            Upgrade.logLogo();
            logger.info("Upgrading SnowMirror.");
            if (args == null || args.length == 0 || args.length > 2) {
                logger.error("Failed to upgrade SnowMirror. Incorrect number of parameters");
                logger.info("Expected parameters: 1. Installation directory (mandatory), 2. Directory with new files (optional).");
                System.exit(2);
            }
            String root = args[0];
            logger.info("SnowMirror path: {}", (Object)root);
            String newFilesDirArg = args.length > 1 ? args[1] : null;
            Path newFilesDir = Upgrade.getNewFilesDir(newFilesDirArg);
            logger.info("Directory with new files: {}", (Object)newFilesDir);
            Path rootDirPath = Path.of(root, new String[0]);
            Path serverXmlPath = Path.of(root + "/conf/server.xml", new String[0]);
            Path sslConfigJsonPath = Path.of(root + "/conf/sslConfig.json", new String[0]);
            Path webXmlPath = Path.of(root + "/conf/web.xml", new String[0]);
            TomcatBinary tomcatBinary = new TomcatBinary(rootDirPath);
            Keystore keystore = new Keystore(rootDirPath);
            Upgrade upgrade = new Upgrade();
            tomcatBinary.deleteAndRetryIfNotAccessible();
            H2Driver h2Driver = new H2Driver(rootDirPath, newFilesDir);
            h2Driver.handle();
            upgrade.backupSelectedFiles(rootDirPath);
            upgrade.deleteOldDirs(rootDirPath);
            upgrade.deleteDirectory(Path.of(root + "/snow-mirror-themes", new String[0]));
            upgrade.cleanAll(Path.of(root + "/conf", new String[0]), List.of(serverXmlPath, webXmlPath, sslConfigJsonPath));
            upgrade.cleanAll(Path.of(root + "/snow-mirror", new String[0]), List.of(Path.of(root + "/snow-mirror/data", new String[0])));
            upgrade.cleanFiles(rootDirPath, List.of("snowMirror.properties", "snowMirror.jks", "snowMirror-https.jks", "uninstall.info"));
            List.of(rootDirPath.resolve("work"), rootDirPath.resolve("webapps")).forEach(upgrade::createEmptyDir);
            upgrade.copyFiles(rootDirPath, newFilesDir);
            new ServerXml(serverXmlPath, sslConfigJsonPath).upgrade();
            new WebXml(webXmlPath).upgrade();
            keystore.importTrustedCertificates();
            keystore.importSnowMirrorJksToCacerts();
            logger.info("Upgrade finished.");
        }
        catch (Exception e) {
            logger.error("Failure.", e);
            System.out.println("Upgrade failed: " + e.getMessage());
            e.printStackTrace();
            System.exit(1);
        }
    }

    private void copyFiles(Path rootDir, Path newVersionDir) throws IOException {
        this.copyDir(rootDir, newVersionDir, "bin");
        this.copyDir(rootDir, newVersionDir, "lib");
        this.copyDir(rootDir, newVersionDir, "conf", pathname -> !"server.xml".equalsIgnoreCase(pathname.getName()) && !"web.xml".equalsIgnoreCase(pathname.getName()));
        this.copyDir(rootDir, newVersionDir, "snow-mirror", pathname -> !"snow-mirror-upgrade.jar".equalsIgnoreCase(pathname.getName()) && !"h2.jar".equalsIgnoreCase(pathname.getName()));
        logger.info("Copying files from {} to {}", (Object)newVersionDir, (Object)rootDir);
        try (Stream<Path> listStream = Files.list(newVersionDir);){
            listStream.filter(x$0 -> Files.isRegularFile(x$0, new LinkOption[0])).forEach(file -> {
                try {
                    Path destPath = rootDir.resolve(file.getFileName());
                    if (!Files.exists(destPath, new LinkOption[0])) {
                        logger.info("Copying {} to {}.", file, (Object)destPath);
                        Files.copy(file, destPath, new CopyOption[0]);
                    } else {
                        logger.info("Not copying {} to {}: it already exists.", file, (Object)destPath);
                    }
                }
                catch (IOException e) {
                    throw new RuntimeException(e);
                }
            });
        }
        logger.info("Finished.");
    }

    private void copyDir(Path rootDir, Path newVersionDir, String dir) throws IOException {
        File srcDir = newVersionDir.resolve(dir).toFile();
        File destDir = rootDir.resolve(dir).toFile();
        logger.info("Copying {} to {}", (Object)srcDir.getAbsolutePath(), (Object)destDir.getAbsolutePath());
        FileUtils.copyDirectory(srcDir, destDir);
        logger.info("Finished.");
    }

    private void copyDir(Path rootDir, Path newVersionDir, String dir, FileFilter filter) throws IOException {
        FileUtils.copyDirectory(newVersionDir.resolve(dir).toFile(), rootDir.resolve(dir).toFile(), filter);
    }

    private void backupSelectedFiles(Path rootDir) throws IOException {
        Path backupDirPath = rootDir.resolve("snow-mirror").resolve("data").resolve("backup");
        Path backupServerXml = backupDirPath.resolve("server.xml");
        Path backupWebXml = backupDirPath.resolve("web.xml");
        FileUtils.forceMkdir(backupDirPath.toFile());
        Map<Path, Path> map = Map.of(rootDir.resolve("conf").resolve("server.xml"), backupServerXml, rootDir.resolve("conf").resolve("web.xml"), backupWebXml, rootDir.resolve("snowMirror.properties"), backupDirPath.resolve("snowMirror.properties"));
        for (Map.Entry<Path, Path> e : map.entrySet()) {
            Files.deleteIfExists(e.getValue());
            Files.copy(e.getKey(), e.getValue(), new CopyOption[0]);
        }
    }

    private void deleteOldDirs(Path rootDir) {
        this.deleteDirectory(rootDir.resolve("bin"));
        this.deleteDirectory(rootDir.resolve("lib"));
        this.deleteDirectory(rootDir.resolve("license"));
        this.deleteDirectory(rootDir.resolve("work"));
        this.deleteDirectory(rootDir.resolve("webapps"));
    }

    private void createEmptyDir(Path dir) {
        try {
            logger.info("Creating directory: {}", (Object)dir);
            Files.createDirectory(dir, new FileAttribute[0]);
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private void deleteDirectory(Path dir) {
        if (!Files.exists(dir, new LinkOption[0])) {
            return;
        }
        logger.info("Deleting directory: {}", (Object)dir);
        try (Stream<Path> pathStream = Files.walk(dir, new FileVisitOption[0]);){
            pathStream.sorted(Comparator.reverseOrder()).map(Path::toFile).forEach(File::delete);
        }
        catch (IOException e) {
            throw new RuntimeException("Failed to delete a directory: " + dir, e);
        }
        logger.info("Finished.");
    }

    private void cleanAll(Path dir, List<Path> exceptions) {
        try {
            logger.info("Deleting all directories and files from '{}' except for: {}", (Object)dir, (Object)exceptions.stream().map(Path::toString).collect(Collectors.joining(", ")));
            this.walkDir(dir, stream -> stream.filter(path -> {
                try {
                    return !Files.isSameFile(path, dir);
                }
                catch (IOException e) {
                    throw new RuntimeException(e);
                }
            }).filter(path -> {
                try {
                    for (Path exceptionPath : exceptions) {
                        if (!Files.exists(exceptionPath, new LinkOption[0]) || !(Files.isDirectory(exceptionPath, new LinkOption[0]) ? path.startsWith(exceptionPath) : Files.isSameFile(path, exceptionPath))) continue;
                        return false;
                    }
                    return true;
                }
                catch (IOException e) {
                    throw new RuntimeException("Failed to delete a directory: " + dir, e);
                }
            }).sorted(Comparator.reverseOrder()).map(Path::toFile).forEach(File::delete));
            logger.info("Finished.");
        }
        catch (IOException e) {
            throw new RuntimeException("Failed to delete a directory: " + dir, e);
        }
    }

    private void walkDir(Path dir, Consumer<Stream<Path>> consumer) throws IOException {
        try (Stream<Path> stream = Files.walk(dir, new FileVisitOption[0]);){
            consumer.accept(stream);
        }
    }

    private void cleanFiles(Path dir, List<String> exceptions) {
        try {
            logger.info("Deleting all files from '{}' except for: {}", (Object)dir, (Object)String.join((CharSequence)", ", exceptions));
            List exceptionPaths = exceptions.stream().map(it -> Path.of(dir + "/" + it, new String[0])).collect(Collectors.toList());
            try (Stream<Path> stream = Files.list(dir);){
                stream.filter(x$0 -> Files.isRegularFile(x$0, new LinkOption[0])).filter(path -> {
                    try {
                        for (Path exceptionPath : exceptionPaths) {
                            if (!Files.exists(exceptionPath, new LinkOption[0]) || !Files.isSameFile(path, exceptionPath)) continue;
                            return false;
                        }
                        return true;
                    }
                    catch (IOException e) {
                        throw new RuntimeException("Failed to clean files in directory: " + dir, e);
                    }
                }).forEach(path -> {
                    try {
                        Files.deleteIfExists(path);
                    }
                    catch (IOException e) {
                        throw new RuntimeException(e);
                    }
                });
            }
            logger.info("Finished.");
        }
        catch (IOException e) {
            throw new RuntimeException("Failed to delete a directory: " + dir, e);
        }
    }

    public static void logLogo() {
        logger.info("{}", (Object)"\n\n   _____                              __  __   _                                        \n  / ____|                            |  \\/  | (_)                                      \n | (___    _ __     ___   __      __ | \\  / |  _   _ __   _ __    ___    _ __          \n  \\___ \\  | '_ \\   / _ \\  \\ \\ /\\ / / | |\\/| | | | | '__| | '__|  / _ \\  | '__| \n  ____) | | | | | | (_) |  \\ V  V /  | |  | | | | | |    | |    | (_) | | |            \n |_____/  |_| |_|  \\___/    \\_/\\_/   |_|  |_| |_| |_|    |_|     \\___/  |_|         \n ______________________________________________________________________________         \n              SnowMirror - Data Replication Tool for ServiceNow                         \n ______________________________________________________________________________         \n");
    }

    private static String getLogPath(String[] args) {
        Object path;
        Object object = path = args != null && args.length >= 1 ? args[0] : "";
        if (!((String)path).endsWith("/") && !((String)path).endsWith("\\")) {
            path = (String)path + "/";
        }
        path = (String)path + "logs/upgrade-tool.log";
        return path;
    }

    private static Path getNewFilesDir(String newFilesDirArg) throws IOException {
        Path newFilesDirPath;
        if (newFilesDirArg != null) {
            newFilesDirPath = Path.of(newFilesDirArg, new String[0]);
        } else {
            Path thisFilePath = Upgrade.getThisFilePath();
            Path installationDirPath = Upgrade.getParent(thisFilePath);
            Path snowMirrorDirPath = Upgrade.getParent(installationDirPath);
            newFilesDirPath = Upgrade.getParent(snowMirrorDirPath);
        }
        Path snowMirrorProperties = newFilesDirPath.resolve("snowMirror.properties");
        logger.info(snowMirrorProperties.toAbsolutePath().toString());
        if (!Files.exists(snowMirrorProperties, new LinkOption[0])) {
            logger.info("Failed to find a directory with new files.");
            System.exit(3);
        }
        return newFilesDirPath;
    }

    public static Path getParent(Path path) {
        Path parent = path.getParent();
        Upgrade.checkPathExists(parent);
        return parent;
    }

    private static void checkPathExists(Path path) {
        if (path == null || !Files.exists(path, new LinkOption[0])) {
            throw new RuntimeException("Failed to find a directory with new files.");
        }
    }

    private static Path getThisFilePath() {
        String path = Upgrade.class.getProtectionDomain().getCodeSource().getLocation().getPath();
        try {
            path = URLDecoder.decode(path, StandardCharsets.UTF_8);
        }
        catch (Exception exception) {
            // empty catch block
        }
        File jarFile = new File(path);
        String absolutePath = jarFile.getAbsolutePath();
        logger.info("This file path: {}", (Object)absolutePath);
        return Path.of(absolutePath, new String[0]);
    }
}

