-- Improving performance of synchronizationView. Note that due to performance reasons the view consists of two parts.
-- The first one is for synchronizations with activityLog and the second for synchronizations without activity log 
-- (i.e. those which have not run yet).

DROP VIEW IF EXISTS synchronizationView;    
CREATE VIEW synchronizationView AS
(
-- select synchronizations which have at least one activity log
SELECT sync.id,
    sync.manual,
    sync.waiting,
    sync.enabled,
    sync.name,
    IF (incrementalScheduler.schedulerDefinitionType = 'MANUALLY' or qrtz_incremental.NEXT_FIRE_TIME = '-1', NULL, qrtz_incremental.NEXT_FIRE_TIME) as nextIncrementalStartLong,
    IF (fullScheduler.schedulerDefinitionType = 'MANUALLY' or qrtz_full.NEXT_FIRE_TIME = '-1', NULL, qrtz_full.NEXT_FIRE_TIME) as nextFullStartLong,
    LEAST(
        COALESCE( IF (incrementalScheduler.schedulerDefinitionType = 'MANUALLY' or qrtz_incremental.NEXT_FIRE_TIME = '-1', NULL, qrtz_incremental.NEXT_FIRE_TIME),
                  IF (fullScheduler.schedulerDefinitionType = 'MANUALLY' or qrtz_full.NEXT_FIRE_TIME = '-1', NULL, qrtz_full.NEXT_FIRE_TIME)
        ),
        COALESCE(IF (fullScheduler.schedulerDefinitionType = 'MANUALLY' or qrtz_full.NEXT_FIRE_TIME = '-1', NULL, qrtz_full.NEXT_FIRE_TIME),
                 IF (incrementalScheduler.schedulerDefinitionType = 'MANUALLY' or qrtz_incremental.NEXT_FIRE_TIME = '-1', NULL, qrtz_incremental.NEXT_FIRE_TIME)
        )
    ) as nextStartLong,
    a.id AS lastLog_id,
    TIMESTAMPDIFF(SECOND, a.startDate, a.endDate) * 1000 AS lastDuration
FROM synchronization AS sync
    JOIN scheduler incrementalScheduler ON sync.inc_scheduler_id = incrementalScheduler.id
    LEFT JOIN scheduler fullScheduler ON sync.full_scheduler_id = fullScheduler.id
    LEFT JOIN QRTZ_TRIGGERS qrtz_incremental ON qrtz_incremental.SCHED_NAME = 'org.springframework.scheduling.quartz.SchedulerFactoryBean#0' and CONCAT('SyncId_', sync.id) = qrtz_incremental.TRIGGER_NAME and qrtz_incremental.TRIGGER_GROUP = 'DEFAULT'
    LEFT JOIN QRTZ_TRIGGERS qrtz_full ON qrtz_full.SCHED_NAME = 'org.springframework.scheduling.quartz.SchedulerFactoryBean#0' and CONCAT('CleanAndSync_SyncId_', sync.id) = qrtz_full.TRIGGER_NAME and qrtz_full.TRIGGER_GROUP = 'DEFAULT',
    activityLog a -- it is much faster to join it using WHERE clause instead of JOIN
WHERE
    -- join the latest activity log. 
    sync.id = a.synchronization AND a.id in (select max(al.id) as maxid from activityLog al group by al.synchronization)
)
UNION
(
-- select synchronizations which have no activity log
SELECT sync.id,
    sync.manual,
    sync.waiting,
    sync.enabled,
    sync.name,
    IF (incrementalScheduler.schedulerDefinitionType = 'MANUALLY' or qrtz_incremental.NEXT_FIRE_TIME = '-1', NULL, qrtz_incremental.NEXT_FIRE_TIME) as nextIncrementalStartLong,
    IF (fullScheduler.schedulerDefinitionType = 'MANUALLY' or qrtz_full.NEXT_FIRE_TIME = '-1', NULL, qrtz_full.NEXT_FIRE_TIME) as nextFullStartLong,
    LEAST(
      COALESCE( IF (incrementalScheduler.schedulerDefinitionType = 'MANUALLY' or qrtz_incremental.NEXT_FIRE_TIME = '-1', NULL, qrtz_incremental.NEXT_FIRE_TIME),
                IF (fullScheduler.schedulerDefinitionType = 'MANUALLY' or qrtz_full.NEXT_FIRE_TIME = '-1', NULL, qrtz_full.NEXT_FIRE_TIME)
      ),
      COALESCE(IF (fullScheduler.schedulerDefinitionType = 'MANUALLY' or qrtz_full.NEXT_FIRE_TIME = '-1', NULL, qrtz_full.NEXT_FIRE_TIME),
               IF (incrementalScheduler.schedulerDefinitionType = 'MANUALLY' or qrtz_incremental.NEXT_FIRE_TIME = '-1', NULL, qrtz_incremental.NEXT_FIRE_TIME)
      )
  ) as nextStartLong,
    null AS lastLog_id,
    null AS lastDuration
FROM synchronization AS sync
    JOIN scheduler incrementalScheduler ON sync.inc_scheduler_id = incrementalScheduler.id
    LEFT JOIN scheduler fullScheduler ON sync.full_scheduler_id = fullScheduler.id
    LEFT JOIN QRTZ_TRIGGERS qrtz_incremental ON qrtz_incremental.SCHED_NAME = 'org.springframework.scheduling.quartz.SchedulerFactoryBean#0' and CONCAT('SyncId_', sync.id) = qrtz_incremental.TRIGGER_NAME and qrtz_incremental.TRIGGER_GROUP = 'DEFAULT'
    LEFT JOIN QRTZ_TRIGGERS qrtz_full ON qrtz_full.SCHED_NAME = 'org.springframework.scheduling.quartz.SchedulerFactoryBean#0' and CONCAT('CleanAndSync_SyncId_', sync.id) = qrtz_full.TRIGGER_NAME and qrtz_full.TRIGGER_GROUP = 'DEFAULT'
WHERE 
    not exists (SELECT al.id FROM activityLog al WHERE al.synchronization = sync.id)
)
    